#!/usr/bin/perl

# $Id: QC40_test.plx,v 1.21 2007/07/26 21:43:54 max Exp $

use warnings;
use strict;
use FileHandle;
use Getopt::Long;
use Pod::Usage;
use Data::Dumper;
use SOAP::Lite soapversion => 1.1;

Getopt::Long::Configure ("bundling_override");
$Data::Dumper::Terse = 1;

# Flags.
my ($help, $man, $verbose, $debug, $no_autoflush);
my ($method, $username, $password, $originator, $deliver_after, $host);
my ($body, $readable, $price, $notification, $uniqueid, $qc50, $campaign_id);
my ($href, $created, $si_expires, $text, $qc60, $quios_sub, $quios_sub_id);
my ($content_type, $port);
# Parameters.
my ($parameter);
# Other variables.
my (@parameter, @flag, $i);
my ($soap, @t, $msisdn, $result, $args_base, $args, $args_extra);
my ($testmode, $type, $class, $udhi, $header, $footer, $set_reply_path);
my ($si_id, $action, @msisdn, @numbers, $elem);

# Flags and parameters.
@parameter =
  (
   '<parameter>' => \$parameter,
  );

@flag =
  (
   'method|me|f=s'             => \$method,
   'username|u=s'              => \$username,
   'password|p=s'              => \$password,
   'host|h=s'                  => \$host,
   'body|b=s'                  => \$body,
   'originator|o=s'            => \$originator,
   'deliver_after|da|a=s'      => \$deliver_after,
   'price|pr|c=s'              => \$price,
   'notification|n=s'          => \$notification,
   'uniqueid|uid|i=s'          => \$uniqueid,
   'qc50|50|5'                 => \$qc50,
   'qc60|60|6'                 => \$qc60,
   'campaign_id|cid|g=i'       => \$campaign_id,
   'href|hr=s'                 => \$href,
   'created|cr=s'              => \$created,
   'si_expires|expires|se|s=s' => \$si_expires,
   'text|txt|tx|t=s'           => \$text,
   'readable|r'                => \$readable,
   'quios_sub|sub|sb=s'        => \$quios_sub,
   'quios_sub_id|sub_id|id=s'  => \$quios_sub_id,
   'content_type|type|ct=s'    => \$content_type,
   'port|po=i'                 => \$port,
   'help|?'                    => \$help,
   'man|m'                     => \$man,
   'verbose|v+'                => \$verbose,
   'debug|d+'                  => \$debug,
   'no_autoflush|na'           => \$no_autoflush,
  );

# Set default values for flags.
$help = $man = $verbose = $debug = $readable = $qc50 = $qc60 = $campaign_id = 0;
$deliver_after = $uniqueid = $quios_sub = $quios_sub_id = $content_type = '';
$method       = 'send_to_number';
$host         = 'soap.ewingz.com';
$username     = 'replace_with_your_username';
$password     = 'replace_with_your_password';
$body         = 'test message';
$originator   = 'replace_with_your_originator';
$price        = 'fteu';
$notification = 'handset';
$href         = 'http://wap.yahoo.com';
$created      = '2007-01-04T00:00:00Z';
$si_expires   = '2007-01-06T00:00:00Z';
$text         = 'Visit yahoo';
$port         = 80;

# Get flags.
GetOptions @flag or pod2usage(2);
pod2usage(1) if $help;
pod2usage(-verbose => 2) if $man;

# Optionally print out flags and values.
print_args( [ @flag ], 'Flags' ) if $debug > 1;

# TODO: Check for illegal values of flags.

# Currently, QC50 does not have all of QC40s methods.  Also
# campaign_id and price are required for the QC50 send_to_number
# methods.

# Get parameters.
pod2usage(2) if 2 * @ARGV != scalar @parameter;
for ( $i = 0; $i < @parameter; $i++ ) {
    ${$parameter[$i]} = shift if ( $i % 2 ); # True if odd.
}
# Optionally print out parameters and values.
print "\n" if $debug > 1;
print_args( [ @parameter ], 'Parameters' ) if $debug > 1;

# Settable stuff

# TODO: Make these variables into flags with the given values as
# default values.
$testmode = 0;
$type = 'GSM0338';
$class = 1;
$udhi = 0;
$header = '';
$footer = '';
$set_reply_path = 0;
$si_id = '';
$action = '';
# End of settable stuff.

# Set derived variables.
if ($method eq 'send_to_number' or
    $method eq 'send_to_number_price' or
    $method eq 'number_lookup' or
    $method eq 'number_lookup_async' or
    $method eq 'wap_push_si' or
    $method eq 'wap_push_sl') {
    $msisdn = $parameter;
} elsif ($method eq 'status' or
         $method eq 'history' or
         $method eq 'cancel') {
    $uniqueid = $parameter;
} elsif ($method eq 'get_messages' or
         $method eq 'carrier_list' or
         $method eq 'coverage_map') {
    $username = $parameter unless $parameter eq 'default';
} elsif ($method eq 'send_to_numbers') {
    @msisdn = split ' ', $parameter;
} else {
    die "Unknown method: $method\n";
}
# End of derived variables.

# Turn off buffering.
unless ( $no_autoflush ) {
    print <<"EOT" if $debug > 1;

Turning on autoflush:
autoflush STDOUT;
autoflush STDERR;

EOT
    autoflush STDOUT; # So we can see our logs as the program runs.
    autoflush STDERR; # So we can see our error logs as the program runs.
}

# Begin real code.
if ($verbose or $debug) {
    # See SOAP::Trace for details.
    # SOAP::Lite->import(+trace => [ 'transport' => \&log_message ]);
    SOAP::Lite->import(+trace => 'debug');
}

# Set up connection.
if ($port == 80) {
    $port = '';
} else {
    $port = ":$port";
}
if ($qc50) {
    $soap = SOAP::Lite->new
      (
       uri      => "http://$host/eWingz/SOAP/QC50",
       proxy    => "http://$host$port/SOAP",
       readable => $readable,
      );
} elsif ($qc60) {
    $soap = SOAP::Lite->new
      (
       uri      => "http://$host/eWingz/SOAP/QC60",
       proxy    => "http://$host$port/SOAP",
       readable => $readable,
      );
} else {
    $soap = SOAP::Lite->new
      (
       uri      => "http://$host/eWingz/SOAP/QC40",
       proxy    => "http://$host$port/SOAP",
       readable => $readable,
      );
}

# Set up arguments for method.
# Arguments common to all methods.
$args_base = <<'EOT';

SOAP::Data->name('username')      ->type('string'  => $username),
SOAP::Data->name('password')      ->type('string'  => $password),
EOT

# Arguments common to all MT methods.
if ($method eq 'send_to_number' or
    $method eq 'send_to_number_price' or
    $method eq 'send_to_numbers' or
    $method eq 'wap_push_si' or
    $method eq 'wap_push_sl') {
    $args = <<'EOT';
SOAP::Data->name('testmode')      ->type('boolean' => $testmode),
SOAP::Data->name('notification')  ->type('string'  => $notification),
EOT
    $args_base .= $args;
}

# Arguments common to most MT methods.
if ($method eq 'send_to_number' or
    $method eq 'send_to_number_price' or
    $method eq 'send_to_numbers') {
    $args = <<'EOT';
SOAP::Data->name('type')          ->type('string'  => $type),
SOAP::Data->name('class')         ->type('integer' => $class),
SOAP::Data->name('udhi')          ->type('boolean' => $udhi),
SOAP::Data->name('originator')    ->type('base64'  => $originator),
SOAP::Data->name('header')        ->type('base64'  => $header),
SOAP::Data->name('body')          ->type('base64'  => $body),
SOAP::Data->name('footer')        ->type('base64'  => $footer),
EOT
    $args_base .= $args;
}

# Arguments specific to given methods.
if ($method eq 'send_to_number' or
    $method eq 'send_to_number_price') {
    $args = <<'EOT';
SOAP::Data->name('msisdn')        ->type('string'  => $msisdn),
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
SOAP::Data->name('set_reply_path')->type('boolean' => $set_reply_path),
SOAP::Data->name('deliver_after') ->type('date'    => $deliver_after),
EOT
    if ($method eq 'send_to_number_price' or $qc50 or $qc60) {
        $args_extra = <<'EOT';
SOAP::Data->name('price')         ->type('string'  => $price),
EOT
        $args .= $args_extra;
    }
    if ($qc50 or $qc60) {
        $args_extra = <<'EOT';
SOAP::Data->name('campaign_id')   ->type('integer' => $campaign_id),
EOT
        $args .= $args_extra;
    }
    if ($qc60) {
        $args_extra = <<'EOT';
SOAP::Data->name('quios_sub')     ->type('string'  => $quios_sub),
SOAP::Data->name('quios_sub_id')  ->type('base64'  => $quios_sub_id),
SOAP::Data->name('content_type')  ->type('string'  => $content_type),
EOT
        $args .= $args_extra;
    }
} elsif ($method eq 'send_to_numbers') {
    for $msisdn (@msisdn) {
        $elem =
          SOAP::Data->name
              ('number' => \SOAP::Data->value
               (
                SOAP::Data->name
                ('msisdn'         => $msisdn)        ->type('string'),
                SOAP::Data->name
                ('uniqueid'       => $uniqueid)      ->type('base64'),
                SOAP::Data->name
                ('set_reply_path' => $set_reply_path)->type('boolean'),
                SOAP::Data->name
                ('deliver_after'  => $deliver_after) ->type('date'),
               )
              )->type("HASH");
        push @numbers, $elem;
    }
    if ($qc50 or $qc60) {
        $args = <<'EOT';
SOAP::Data->name('price')         ->type('string'  => $price),
SOAP::Data->name('campaign_id')   ->type('integer' => $campaign_id),
SOAP::Data->name("numbers")       ->type("ARRAY"   => \@numbers),
EOT
	if ($qc60) {
	    $args_extra = <<'EOT';
SOAP::Data->name('quios_sub')     ->type('string'  => $quios_sub),
SOAP::Data->name('quios_sub_id')  ->type('base64'  => $quios_sub_id),
SOAP::Data->name('content_type')  ->type('string'  => $content_type),
EOT
	    $args .= $args_extra;
	}
    } else {
        $args = <<'EOT';
SOAP::Data->name("numbers")       ->type("ARRAY"   => \@numbers),
EOT
    }
} elsif ($method eq 'wap_push_si' or
         $method eq 'wap_push_sl') {
    if ($qc50) {
        $args = <<'EOT';
SOAP::Data->name('msisdn')        ->type('string'  => $msisdn),
SOAP::Data->name('href')          ->type('string'  => $href),
SOAP::Data->name('campaign_id')   ->type('integer' => $campaign_id),
EOT
        if ($method eq 'wap_push_si') {
            $args_extra = <<'EOT';
SOAP::Data->name('created')       ->type('string'  => $created),
SOAP::Data->name('si_expires')    ->type('string'  => $si_expires),
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
SOAP::Data->name('action')        ->type('string'  => $action),
SOAP::Data->name('si_id')         ->type('string'  => $si_id),
SOAP::Data->name('text')          ->type('string'  => $text),
EOT
        } elsif ($method eq 'wap_push_sl') {
            $args_extra = <<'EOT';
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
SOAP::Data->name('action')        ->type('string'  => $action),
EOT
        } else {
            die "Unhandled method: $method";
        }
        $args .= $args_extra;
    } elsif ($qc60) {
        $args = <<'EOT';
SOAP::Data->name('originator')    ->type('base64'  => $originator),
SOAP::Data->name('msisdn')        ->type('string'  => $msisdn),
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
SOAP::Data->name('deliver_after') ->type('date'    => $deliver_after),
SOAP::Data->name('price')         ->type('string'  => $price),
SOAP::Data->name('campaign_id')   ->type('integer' => $campaign_id),
SOAP::Data->name('quios_sub')     ->type('string'  => $quios_sub),
SOAP::Data->name('quios_sub_id')  ->type('base64'  => $quios_sub_id),
SOAP::Data->name('content_type')  ->type('string'  => $content_type),
SOAP::Data->name('href')          ->type('string'  => $href),
SOAP::Data->name('action')        ->type('string'  => $action),
EOT
        if ($method eq 'wap_push_si') {
            $args_extra = <<'EOT';
SOAP::Data->name('created')       ->type('string'  => $created),
SOAP::Data->name('si_expires')    ->type('string'  => $si_expires),
SOAP::Data->name('si_id')         ->type('string'  => $si_id),
SOAP::Data->name('text')          ->type('string'  => $text),
EOT
        } elsif ($method eq 'wap_push_sl') {
            $args_extra = '';
        } else {
            die "Unhandled method: $method";
        }
        $args .= $args_extra;
    } else {
        $args = <<'EOT';
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
SOAP::Data->name('msisdn')        ->type('string'  => $msisdn),
SOAP::Data->name('href')          ->type('string'  => $href),
SOAP::Data->name('action')        ->type('string'  => $action),
EOT
        if ($method eq 'wap_push_si') {
            $args_extra = <<'EOT';
SOAP::Data->name('si_id')         ->type('string'  => $si_id),
SOAP::Data->name('created')       ->type('string'  => $created),
SOAP::Data->name('si_expires')    ->type('string'  => $si_expires),
SOAP::Data->name('text')          ->type('string'  => $text),
SOAP::Data->name('originator')    ->type('base64'  => $originator),
EOT
        } elsif ($method eq 'wap_push_sl') {
            $args_extra = '';
        } else {
            die "Unhandled method: $method";
        }
        $args .= $args_extra;
    }
} elsif ($method eq 'status' or
         $method eq 'history' or
         $method eq 'cancel') {
    $args = <<'EOT';
SOAP::Data->name('uniqueid')      ->type('base64'  => $uniqueid),
EOT
} elsif ($method eq 'number_lookup' or
         $method eq 'number_lookup_async' ) {
    $args = <<'EOT';
SOAP::Data->name('msisdn')        ->type('string'  => $msisdn),
EOT
} else {
    $args = '';
}

$args = $args_base . $args;

# Print out dynamically constructed SOAP::Lite method request and
# arguments.
print "\$result = \$soap->$method($args)->result\n" if $debug > 1;

# Dynamically construct SOAP arguments and call method.
eval "\$result = \$soap->\$method($args)->result";

# Print out result.
print "$method result:\n";

print Dumper($result),"\n";

exit;

# $debug, $verbose, and $execute are visible to these sub routines
# so we do not pass them as arguments.

# Print out parameters and flags.
sub print_args {
    my ($ar, $heading) = @_;
    my ($i);

    print "$heading:\n";
    $i = 0;
    for (@$ar) {
        if ($i++ % 2) {         # True if odd.
            print "$$_\n";
        } else {
            print "$_ => ";
        }
    }
}

# Experimental.  See SOAP::Lite->import above.
sub log_message {
    print Dumper(@_),"\n";
}

__END__

=head1 NAME

B<QC40_test.plx> - A script to make QC40 SOAP calls.

=head1 SYNOPSIS

 QC40_test.plx <msisdn|'msisdn1 msisdn2...msisdnN'|uniqueid|username>

 Options:                                Default Value
 --method, -me, -f                       "send_to_number"
 --username, -u                          "replace_with_your_username"
 --password, -p                          "replace_with_your_password"
 --host, -h                              "soap.ewingz.com"
 --body, -b                              "test message"
 --originator, -o                        "replace_with_your_originator"
 --deliver_after, --da, -a               ""
 --price, --pr, -c                       "fteu"
 --notification, -n                      "handset"
 --uniqueid, --uid, -i                   ""
 --qc50, --50, -5'
 --qc60, --60, -6'
 --campaign_id, --cid, -g                0
 --href, --hr                            "http://wap.yahoo.com"
 --created, --cr                         "2007-01-04T00:00:00Z"
 --si_expires, --expires, --se, -s       "2007-01-06T00:00:00Z"
 --text, --txt, --tx, -t                 "Visit yahoo"
 --readable, -r
 --port --po                             80
 --help, -?
 --man, -m
 --verbose, -v
 --debug, -d
 --no_autoflush, -b

=head1 OPTIONS

=over 8

=item B<--method>

Method to invoke.  Supported methods are send_to_number,
send_to_number_price, send_to_numbers, wap_push_si, wap_push_sl,
status, history, get_messages, cancel, number_lookup,
number_lookup_async, carrier_list, and coverage_map.

The parameter to QC40_test.plx is interpreted as an msisdn for
send_to_number, send_to_number_price, wap_push_si, wap_push_sl,
number_lookup, and number_lookup_async; as a space separated list of
msisdns by send_to_numbers (use quotes to pass as a single argument);
as a uniqueid for status, history, and cancel; and as a username for
get_messages, carrier_list, and coverage_map.

When parameter is interpreted as a username the special value
"default" will cause the value of the flag B<--username> to be used.

The default value is B<send_to_number>.

=item B<--username>

eWingz account to authenticate.

The default value is B<username>.

=item B<--password>

Password for the eWingz account.

The default value is B<password>.

=item B<--host>

Host running SOAP server.

The default value is B<soap.ewingz.com>.

=item B<--body>

Message body for the send_to_number methods.

The default value is B<test message>.

=item B<--originator>

Originator for the send_to_number methods.

The default value is B<originator>.

=item B<--deliver_after>

Deliver_After for the send_to_number methods.

The format of the date is YYYY-MM-DD HH24:MI:SS.

The default value is the the empty string.

=item B<--price>

Price for the send_to_number_price method.

The default value is B<fteu>.

=item B<--notification>

Notification for the send_to_number methods.

The default value is B<handset>.

=item B<--uniqueid>

Uniqueid for the send_to_number methods.

The default value is the empty string.

=item B<--qc50>

Use QC50 interface instead of the QC40 interface.

=item B<--qc60>

Use QC60 interface instead of the QC40 interface.

=item B<--campaign_id>

Campaign_Id for the QC50 and QC60 send methods.

The default value is 0 which is a non-existent campaign_id.

=item B<--href>

href for the wap_push_si and wap_push_sl methods.

The default value is "http://wap.yahoo.com".

=item B<--created>

created for the wap_push_si and wap_push_sl methods.

The default value is "2007-01-04T00:00:00Z".

=item B<--si_expires>

si_expires for the wap_push_si and wap_push_sl methods.

The default value is "2007-01-06T00:00:00Z".

=item B<--text>

text for the wap_push_si and wap_push_sl methods.

The default value is "Visit yahoo".

=item B<--readable>

Make the request XML, which is printed when the B<--verbose> or
B<--debug> flags are used, more human readable.

=item B<--port>

Specify the HTTP port.

The default value is 80.

=item B<--help>

=item B<--man>

=item B<--verbose>

=item B<--debug>

Can be repeated for more output.

=item B<--no_autoflush>

Buffer output.

=back

=head1 DESCRIPTION

B<QC40_test.plx> tests the various Q-Caster 4.0 SOAP RPC calls.  It
can also be used to test the Q-Caster 5.0 and 6.0 SOAP RPC calls.

=head1 BUGS

Not all the send_to_number and wap_push methods' parameters are
settable via a flag.

Needs more input checking when the qc50 or qc60 flag is used to avoid
using non-existent QC50 or QC60 methods.

Needs more flag consistency checking to avoid supplying flags not used
by a method, or failure to supply a flag needed by a method, e.g.,
campaign_id when the qc50 or qc60 flag is supplied.

=head1 INFO

B<Author> Maximilian Mohun B<Date> November 12, 2005

$Id: QC40_test.plx,v 1.21 2007/07/26 21:43:54 max Exp $

=cut
